<?php
/* --------------------------------------------------------------
   AdminErrorHandler.php 2020-05-29
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Application\ErrorHandling;

use Exception;
use Gambio\Core\Logging\LoggerBuilder;
use Psr\Log\LoggerInterface;

/**
 * Class AdminErrorHandler
 *
 * @package Gambio\Admin\Application\ErrorHandling
 */
class AdminErrorHandler
{
    /**
     * @var LoggerInterface
     */
    private $logger;
    
    /**
     * @var string
     */
    private $logHash;
    
    
    public function __construct(LoggerBuilder $loggerBuilder)
    {
        $this->logger  = $loggerBuilder->changeNamespace('admin')->omitRequestData()->build();
        $this->logHash = uniqid('logger-', true);
    }
    
    
    /**
     * Logs occurring errors.
     *
     * @param int    $errno
     * @param string $errstr
     * @param string $errfile
     * @param int    $errline
     * @param array  $errcontext
     *
     * @return bool
     */
    public function __invoke(int $errno, string $errstr, string $errfile, int $errline, array $errcontext): bool
    {
        switch ($errno) {
            // handling notices
            case E_USER_NOTICE:
            case E_NOTICE:
                $this->logNotice($errno, $errstr, $errfile, $errline, $errcontext);
                break;
            
            // handling errors
            case E_ERROR:
            case E_PARSE:
            case E_CORE_ERROR:
            case E_COMPILE_ERROR:
            case E_RECOVERABLE_ERROR:
            case E_USER_ERROR:
                $this->logError($errno, $errstr, $errfile, $errline, $errcontext);
                throw new Exception('A critical error occurred and had been logged with hash "' . $this->logHash
                                    . '".');
            
            // handling warnings
            case E_COMPILE_WARNING:
            case E_WARNING:
            case E_USER_WARNING:
            case E_CORE_WARNING:
            case E_STRICT:
            case E_USER_DEPRECATED:
            case E_DEPRECATED:
            default:
                $this->logWarning($errno, $errstr, $errfile, $errline, $errcontext);
        }
        
        return true;
    }
    
    
    /**
     * @param int    $errno
     * @param string $errstr
     * @param string $errfile
     * @param int    $errline
     * @param array  $errcontext
     */
    private function logNotice(int $errno, string $errstr, string $errfile, int $errline, array $errcontext): void
    {
        // currently we are not handling notices, because there are too many :(
    }
    
    
    /**
     * @param int    $errno
     * @param string $errstr
     * @param string $errfile
     * @param int    $errline
     * @param array  $errcontext
     */
    private function logError(int $errno, string $errstr, string $errfile, int $errline, array $errcontext): void
    {
        $obCleanCache = @ob_get_clean();
        $this->logger->error($errstr,
                             [
                                 'code'         => $errno,
                                 'file'         => $errfile,
                                 'line'         => $errline,
                                 'content'      => $errcontext,
                                 'obCleanCache' => $obCleanCache,
                                 'logHash'      => $this->logHash,
                             ]);
    }
    
    
    /**
     * @param int    $errno
     * @param string $errstr
     * @param string $errfile
     * @param int    $errline
     * @param array  $errcontext
     */
    private function logWarning(int $errno, string $errstr, string $errfile, int $errline, array $errcontext): void
    {
        $this->logger->warning($errstr,
                               [
                                   'code'    => $errno,
                                   'file'    => $errfile,
                                   'line'    => $errline,
                                   'content' => $errcontext,
                                   'logHash' => $this->logHash,
                               ]);
    }
}